<?php
require_once 'config.php';
require 'vendor/autoload.php';

use Slim\Factory\AppFactory;
use phpseclib3\Crypt\PublicKeyLoader;
use phpseclib3\Crypt\PKCS1;


function parse_json($prefix, $obj, &$result) {
    if (is_array($obj)) {
        foreach ($obj as $key => $value) {
            $new_prefix = $prefix ? "{$prefix}:{$key}" : $key;
            parse_json($new_prefix, $value, $result);
        }
    } else {
        if ($obj) {
            $result[] = "{$prefix}:{$obj}";
        } else {
            $result[] = "{$prefix}:None";
        }
    }
}

function normalize_message($payload): string {
    $result = [];
    parse_json('', $payload, $result);
    sort($result);
    return implode(';', $result);
}

$app = AppFactory::create();


$app->post('/callback/{status}', function ($request, $response, $args) use ($public_key_file_path) {
    $headers = $request->getHeaders();
    $body = (string) $request->getBody();

    // Логируем заголовки в виде JSON
    error_log("Headers: " . json_encode($headers));

    // Логируем тело
    error_log("Body: " . $body);

    if (!$body) {
        error_log('Received an empty body.');
        return $response->withStatus(409);
    }

    $payload = json_decode($body, true);
    if (!$payload) {
        error_log('Invalid JSON payload: ' . $body);
        return $response->withStatus(409);
    }

    $joined_result = normalize_message($payload);

    $timestamp = $headers['X-Access-Timestamp'][0] ?? null;
    if (!$timestamp) {
        error_log('Missing X-Access-Timestamp header.');
        return $response->withStatus(409);
    }

    $message = strtr(base64_encode($joined_result) . $timestamp, '+/', '-_');

    $access_token = $headers['X-Access-Token'][0] ?? null;
    if (!$access_token) {
        error_log('Missing X-Access-Token header.');
        return $response->withStatus(409);
    }

    try {
        $raw_public_key = file_get_contents($public_key_file_path);

        $public_key_loaded = PublicKeyLoader::load($raw_public_key);
        $public_key_string = $public_key_loaded->toString('PKCS1');
    } catch (\Exception $e) {
        error_log('Failed to load public key: ' . $e->getMessage());
        return $response->withStatus(409);
    }

    try {
        $access_token_decoded = base64_decode($access_token);

        $access_token_loaded = PublicKeyLoader::load($access_token_decoded);
        $access_token_string = $access_token_loaded->toString('PKCS1');
    } catch (\Exception $e) {
        error_log('Failed to process access token as public key: ' . $e->getMessage());
        return $response->withStatus(409);
    }

    $public_key_string = str_replace(["\r\n", "\r"], "\n", $public_key_string);
    $access_token_string = str_replace(["\r\n", "\r"], "\n", $access_token_string);

    $new_key = base64_encode($public_key_string);
    $new_access_token = base64_encode($access_token_string);


    if ($new_key !== $new_access_token) {
        error_log("Public keys do not match!");
        error_log("Decoded token key:\n{$new_access_token}");
        error_log("File key:\n{$new_key}");
        return $response->withStatus(409);
    }


    $signature = $headers['X-Access-Signature'][0] ?? null;
    if (!$signature) {
        error_log('Missing X-Access-Signature header.');
        return $response->withStatus(409);
    }

    try {

        $raw_public_key = file_get_contents($public_key_file_path);
        $public_key_resource = openssl_pkey_get_public($raw_public_key);
        if (!$public_key_resource) {
            throw new Exception("Invalid public key.");
        }

        $signature_decoded = base64_decode(strtr($signature, '-_', '+/'));
        if (!$signature_decoded) {
            throw new Exception("Failed to decode signature.");
        }

        $result = openssl_verify($message, $signature_decoded, $public_key_resource, OPENSSL_ALGO_SHA256);

        if ($result === 1) {
            error_log("Signature is valid (via OpenSSL).");
        } elseif ($result === 0) {
            error_log("Invalid signature (via OpenSSL).");
        } else {
            throw new Exception("OpenSSL verification error.");
        }

    } catch (\Exception $e) {
        error_log('Error with OpenSSL: ' . $e->getMessage());
        return $response->withStatus(409);
    }

    return $response->withStatus(200);
});

$app->run();